<?php
require_once __DIR__ . '/../config/env.php';

class JWTHandler {
    private $secret;
    
    public function __construct() {
        $this->secret = JWT_SECRET;
    }
    
    public function generateToken($payload, $expiresIn = 7200) { // 2 hours default
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload['iat'] = time();
        $payload['exp'] = time() + $expiresIn;
        
        $payload = json_encode($payload);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, $this->secret, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        return $base64Header . "." . $base64Payload . "." . $base64Signature;
    }
    
    public function verifyToken($token) {
        try {
            $parts = explode('.', $token);
            
            if (count($parts) !== 3) {
                return false;
            }
            
            [$header, $payload, $signature] = $parts;
            
            // Verify signature
            $expectedSignature = hash_hmac('sha256', $header . "." . $payload, $this->secret, true);
            $expectedSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($expectedSignature));
            
            if (!hash_equals($signature, $expectedSignature)) {
                return false;
            }
            
            // Decode payload
            $payload = str_replace(['-', '_'], ['+', '/'], $payload);
            $payload = base64_decode($payload . str_repeat('=', (4 - strlen($payload) % 4) % 4));
            $payloadData = json_decode($payload, true);
            
            // Check expiration
            if (isset($payloadData['exp']) && $payloadData['exp'] < time()) {
                return false;
            }
            
            return $payloadData;
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function getTokenFromHeader() {
        $headers = getallheaders();
        $authorization = $headers['Authorization'] ?? '';
        
        if (preg_match('/Bearer\s(\S+)/', $authorization, $matches)) {
            return $matches[1];
        }
        
        return null;
    }
}
?>
