<?php
/**
 * Infinity Medical Installation Wizard
 * Complete Auto-Installation System
 */
session_start();

// Check if already installed
if (file_exists('../api/config/installed.lock')) {
    header('Location: ../index.html');
    exit();
}

$step = $_GET['step'] ?? 1;
$error = '';
$success = '';

// Handle form submissions
if ($_POST) {
    switch ($step) {
        case 2:
            // Database connection test
            $result = testDatabaseConnection($_POST);
            if ($result['success']) {
                $_SESSION['db_config'] = $_POST;
                header('Location: ?step=3');
                exit();
            } else {
                $error = $result['error'];
            }
            break;
            
        case 3:
            // Admin setup
            $_SESSION['admin_config'] = $_POST;
            header('Location: ?step=4');
            exit();
            break;
            
        case 4:
            // Final installation
            $result = performInstallation();
            if ($result['success']) {
                $success = 'Installation completed successfully!';
            } else {
                $error = $result['error'];
            }
            break;
    }
}

function testDatabaseConnection($config) {
    try {
        $dsn = "mysql:host={$config['db_host']};dbname={$config['db_name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return ['success' => true];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => 'Database connection failed: ' . $e->getMessage()];
    }
}

function performInstallation() {
    try {
        // 1. Create configuration file
        createConfigFile();
        
        // 2. Create database tables
        createDatabaseTables();
        
        // 3. Create lock file
        file_put_contents('../api/config/installed.lock', date('Y-m-d H:i:s'));
        
        // 4. Redirect to main site
        header('Refresh: 3; URL=../index.html');
        
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function createConfigFile() {
    $db_config = $_SESSION['db_config'];
    $admin_config = $_SESSION['admin_config'];
    
    $config_content = "<?php
// Auto-generated configuration file
// Created: " . date('Y-m-d H:i:s') . "

// Database Configuration
define('DB_HOST', '{$db_config['db_host']}');
define('DB_NAME', '{$db_config['db_name']}');
define('DB_USER', '{$db_config['db_user']}');
define('DB_PASS', '{$db_config['db_pass']}');

// Admin Credentials
define('ADMIN_USERNAME', '{$admin_config['admin_user']}');
define('ADMIN_PASSWORD', '{$admin_config['admin_pass']}');

// JWT Configuration
define('JWT_SECRET', '" . bin2hex(random_bytes(32)) . "');

// Email Configuration
define('SMTP_HOST', '{$admin_config['smtp_host']}');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', '{$admin_config['smtp_user']}');
define('SMTP_PASSWORD', '{$admin_config['smtp_pass']}');
define('SMTP_FROM_EMAIL', '{$admin_config['smtp_user']}');
define('SMTP_FROM_NAME', 'Promed Solutions Team');

// File Upload Configuration
define('UPLOAD_PATH', __DIR__ . '/../uploads/');
define('UPLOAD_URL', '/api/uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_EXTENSIONS', ['jpg', 'jpeg', 'png', 'gif', 'webp']);
?>";
    
    file_put_contents('../api/config/env.php', $config_content);
}

function createDatabaseTables() {
    $db_config = $_SESSION['db_config'];
    
    $dsn = "mysql:host={$db_config['db_host']};dbname={$db_config['db_name']};charset=utf8mb4";
    $pdo = new PDO($dsn, $db_config['db_user'], $db_config['db_pass']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Read SQL file and execute
    $sql = file_get_contents('database.sql');
    $pdo->exec($sql);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Infinity Medical - Installation Wizard</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        .step-active { @apply bg-blue-600 text-white; }
        .step-completed { @apply bg-green-500 text-white; }
        .step-inactive { @apply bg-gray-300 text-gray-600; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="min-h-screen flex items-center justify-center py-12 px-4">
        <div class="max-w-4xl w-full space-y-8">
            <!-- Header -->
            <div class="text-center">
                <h1 class="text-3xl font-bold text-gray-900 mb-2">🏥 Infinity Medical</h1>
                <p class="text-xl text-gray-600">Installation Wizard</p>
            </div>

            <!-- Progress Steps -->
            <div class="flex justify-center mb-8">
                <div class="flex items-center space-x-4">
                    <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium <?= $step >= 1 ? 'step-active' : 'step-inactive' ?>">1</div>
                        <span class="ml-2 text-sm font-medium text-gray-500">Welcome</span>
                    </div>
                    <div class="w-8 h-0.5 bg-gray-300"></div>
                    <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium <?= $step >= 2 ? ($step > 2 ? 'step-completed' : 'step-active') : 'step-inactive' ?>">2</div>
                        <span class="ml-2 text-sm font-medium text-gray-500">Database</span>
                    </div>
                    <div class="w-8 h-0.5 bg-gray-300"></div>
                    <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium <?= $step >= 3 ? ($step > 3 ? 'step-completed' : 'step-active') : 'step-inactive' ?>">3</div>
                        <span class="ml-2 text-sm font-medium text-gray-500">Admin Setup</span>
                    </div>
                    <div class="w-8 h-0.5 bg-gray-300"></div>
                    <div class="flex items-center">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium <?= $step >= 4 ? 'step-active' : 'step-inactive' ?>">4</div>
                        <span class="ml-2 text-sm font-medium text-gray-500">Complete</span>
                    </div>
                </div>
            </div>

            <!-- Installation Steps -->
            <div class="bg-white rounded-lg shadow-md p-8">
                <?php if ($error): ?>
                    <div class="mb-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded">
                        ❌ <?= htmlspecialchars($error) ?>
                    </div>
                <?php endif; ?>

                <?php if ($success): ?>
                    <div class="mb-4 p-4 bg-green-100 border border-green-400 text-green-700 rounded">
                        ✅ <?= htmlspecialchars($success) ?>
                        <br>Redirecting to your website in 3 seconds...
                    </div>
                <?php endif; ?>

                <?php switch($step): case 1: ?>
                    <!-- Step 1: Welcome -->
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-gray-900 mb-4">Welcome to Infinity Medical</h2>
                        <p class="text-gray-600 mb-6">
                            This wizard will help you set up your medical equipment website in just a few steps.
                        </p>
                        
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-6 mb-6">
                            <h3 class="font-semibold text-blue-900 mb-2">✅ What You'll Get:</h3>
                            <ul class="text-blue-800 text-left space-y-2">
                                <li>• Complete medical equipment catalog</li>
                                <li>• Contact forms with email notifications</li>
                                <li>• Rental service management</li>
                                <li>• Admin panel for parts management</li>
                                <li>• Responsive design for all devices</li>
                            </ul>
                        </div>

                        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                            <p class="text-yellow-800">
                                <strong>📋 Requirements:</strong> PHP 7.4+, MySQL 5.7+, mod_rewrite enabled
                            </p>
                        </div>

                        <a href="?step=2" class="inline-block bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 font-medium">
                            Start Installation →
                        </a>
                    </div>
                    <?php break; case 2: ?>
                    
                    <!-- Step 2: Database Setup -->
                    <h2 class="text-2xl font-bold text-gray-900 mb-4">📊 Database Configuration</h2>
                    <p class="text-gray-600 mb-6">Enter your cPanel MySQL database credentials:</p>
                    
                    <form method="POST" class="space-y-4">
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Database Host</label>
                                <input type="text" name="db_host" value="localhost" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Database Name</label>
                                <input type="text" name="db_name" placeholder="your_username_infinity" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Database Username</label>
                                <input type="text" name="db_user" placeholder="your_username_dbuser" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Database Password</label>
                                <input type="password" name="db_pass" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                        
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                            <p class="text-blue-800 text-sm">
                                <strong>💡 Tip:</strong> You can find these details in your cPanel → MySQL Databases section.
                            </p>
                        </div>
                        
                        <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 font-medium">
                            Test Connection →
                        </button>
                    </form>
                    <?php break; case 3: ?>
                    
                    <!-- Step 3: Admin Setup -->
                    <h2 class="text-2xl font-bold text-gray-900 mb-4">👤 Admin & Email Setup</h2>
                    <p class="text-gray-600 mb-6">Create your admin account and configure email settings:</p>
                    
                    <form method="POST" class="space-y-6">
                        <div class="bg-gray-50 p-4 rounded-lg">
                            <h3 class="font-semibold text-gray-900 mb-3">Admin Account</h3>
                            <div class="grid grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Admin Username</label>
                                    <input type="text" name="admin_user" value="admin" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Admin Password</label>
                                    <input type="password" name="admin_pass" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                </div>
                            </div>
                        </div>
                        
                        <div class="bg-gray-50 p-4 rounded-lg">
                            <h3 class="font-semibold text-gray-900 mb-3">Email Configuration</h3>
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">SMTP Host</label>
                                    <input type="text" name="smtp_host" value="mail.infinitymedical.co" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                </div>
                                <div class="grid grid-cols-2 gap-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                        <input type="email" name="smtp_user" placeholder="noreply@infinitymedical.co" required
                                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Password</label>
                                        <input type="password" name="smtp_pass" required
                                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 font-medium">
                            Continue to Installation →
                        </button>
                    </form>
                    <?php break; case 4: ?>
                    
                    <!-- Step 4: Installation Complete -->
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-gray-900 mb-4">🎉 Installation Complete!</h2>
                        
                        <?php if (!$success): ?>
                            <form method="POST">
                                <p class="text-gray-600 mb-6">Ready to install your Infinity Medical website?</p>
                                <button type="submit" class="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 font-medium">
                                    🚀 Complete Installation
                                </button>
                            </form>
                        <?php else: ?>
                            <div class="bg-green-50 border border-green-200 rounded-lg p-6 mb-6">
                                <h3 class="font-semibold text-green-900 mb-2">✅ Installation Successful!</h3>
                                <p class="text-green-800">Your website is ready and all systems are configured.</p>
                            </div>
                            
                            <div class="grid grid-cols-2 gap-4 text-left">
                                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                    <h4 class="font-semibold text-blue-900 mb-2">🌐 Your Website</h4>
                                    <a href="../index.html" class="text-blue-600 hover:underline">https://infinitymedical.co</a>
                                </div>
                                <div class="bg-purple-50 border border-purple-200 rounded-lg p-4">
                                    <h4 class="font-semibold text-purple-900 mb-2">⚙️ Admin Panel</h4>
                                    <a href="../admin" class="text-purple-600 hover:underline">Login with your credentials</a>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                    <?php break; endswitch; ?>
            </div>
            
            <!-- Footer -->
            <div class="text-center text-gray-500 text-sm">
                <p>Infinity Medical Installation Wizard v1.0</p>
            </div>
        </div>
    </div>
</body>
</html>
