<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/EmailSender.php';

class RentalController {
    private $db;
    private $emailSender;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
        $this->emailSender = new EmailSender();
    }
    
    public function handleRentalForm() {
        try {
            // Get JSON input
            $input = json_decode(file_get_contents('php://input'), true);
            
            // Validate required fields
            $requiredFields = ['firstName', 'lastName', 'rentalService'];
            foreach ($requiredFields as $field) {
                if (empty($input[$field])) {
                    http_response_code(400);
                    echo json_encode(['message' => "Field '$field' is required"]);
                    return;
                }
            }
            
            // Handle rentalService - it might be an object with value property
            $rentalService = $input['rentalService'];
            if (is_array($rentalService) && isset($rentalService['value'])) {
                $rentalService = $rentalService['value'];
            }
            
            // Prepare data for database
            $data = [
                'first_name' => $input['firstName'],
                'last_name' => $input['lastName'],
                'company_name' => $input['companyName'] ?? '',
                'email' => $input['email'] ?? '',
                'phone_number' => $input['phoneNumber'] ?? '',
                'rental_service' => $rentalService,
                'part_description' => $input['partDescription'] ?? ''
            ];
            
            // Insert into database
            $stmt = $this->db->prepare("
                INSERT INTO rental 
                (first_name, last_name, company_name, email, phone_number, rental_service, part_description) 
                VALUES (:first_name, :last_name, :company_name, :email, :phone_number, :rental_service, :part_description)
            ");
            
            $stmt->execute($data);
            
            // Send email notifications
            $this->emailSender->sendFormNotification($data, 'rental');
            
            // Return success response
            http_response_code(201);
            echo json_encode([
                'message' => 'Form submitted successfully',
                'data' => array_merge($data, ['id' => $this->db->lastInsertId()])
            ]);
            
        } catch (Exception $e) {
            error_log('Rental form submission error: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode(['message' => 'Server error while submitting form']);
        }
    }
}
?>
