<?php
/**
 * Infinity Medical - Professional Installation Script
 * Complete automated installation system
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if already installed
$lockFile = __DIR__ . '/api/config/installed.lock';
if (file_exists($lockFile) && !isset($_GET['reinstall'])) {
    die('
    <!DOCTYPE html>
    <html>
    <head>
        <title>Already Installed</title>
        <style>
            body { font-family: Arial, sans-serif; text-align: center; padding: 50px; }
            .box { max-width: 600px; margin: 0 auto; padding: 30px; border: 2px solid #4CAF50; border-radius: 10px; }
            .btn { display: inline-block; padding: 10px 20px; background: #4CAF50; color: white; text-decoration: none; border-radius: 5px; margin: 10px; }
        </style>
    </head>
    <body>
        <div class="box">
            <h2>✅ Installation Already Complete</h2>
            <p>Your Infinity Medical website is already installed.</p>
            <a href="index.html" class="btn">Visit Website</a>
            <a href="?reinstall=1" class="btn" style="background: #f44336;">Reinstall</a>
        </div>
    </body>
    </html>
    ');
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$error = '';
$success = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        case 2:
            // Test database connection
            $result = testDatabaseConnection($_POST);
            if ($result['success']) {
                $_SESSION['db_config'] = $_POST;
                header('Location: ?step=3');
                exit();
            } else {
                $error = $result['error'];
            }
            break;
            
        case 3:
            // Admin setup
            $_SESSION['admin_config'] = $_POST;
            header('Location: ?step=4');
            exit();
            break;
            
        case 4:
            // Final installation
            $result = performInstallation();
            if ($result['success']) {
                $success = 'Installation completed successfully!';
                // Create lock file
                @file_put_contents($lockFile, date('Y-m-d H:i:s'));
            } else {
                $error = $result['error'];
            }
            break;
    }
}

function testDatabaseConnection($config) {
    try {
        $dsn = "mysql:host={$config['db_host']};charset=utf8mb4";
        $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Try to create database if it doesn't exist
        try {
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$config['db_name']}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        } catch (PDOException $e) {
            // Database might already exist, continue
        }
        
        // Test connection to the database
        $dsn = "mysql:host={$config['db_host']};dbname={$config['db_name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        return ['success' => true];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => 'Database connection failed: ' . $e->getMessage()];
    }
}

function performInstallation() {
    try {
        // 1. Create configuration file
        createConfigFile();
        
        // 2. Create database tables
        createDatabaseTables();
        
        // 3. Create uploads directory
        $uploadDir = __DIR__ . '/api/uploads';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // 4. Create .htaccess for uploads
        $htaccess = $uploadDir . '/.htaccess';
        if (!file_exists($htaccess)) {
            file_put_contents($htaccess, "Options -Indexes\n<FilesMatch \"\\.(php|php3|php4|php5|phtml|pl|py|jsp|asp|sh|cgi)$\">\n    Order Allow,Deny\n    Deny from all\n</FilesMatch>");
        }
        
        return ['success' => true];
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function createConfigFile() {
    $db_config = $_SESSION['db_config'];
    $admin_config = $_SESSION['admin_config'];
    
    $config_content = "<?php
// Infinity Medical Configuration File
// Auto-generated on " . date('Y-m-d H:i:s') . "
// DO NOT EDIT MANUALLY - Use install.php to reconfigure

// Database Configuration
define('DB_HOST', '" . addslashes($db_config['db_host']) . "');
define('DB_NAME', '" . addslashes($db_config['db_name']) . "');
define('DB_USER', '" . addslashes($db_config['db_user']) . "');
define('DB_PASS', '" . addslashes($db_config['db_pass']) . "');

// Admin Credentials
define('ADMIN_USERNAME', '" . addslashes($admin_config['admin_user']) . "');
define('ADMIN_PASSWORD', '" . addslashes($admin_config['admin_pass']) . "');

// JWT Configuration
define('JWT_SECRET', '" . bin2hex(random_bytes(32)) . "');

// Email Configuration
define('SMTP_HOST', '" . addslashes($admin_config['smtp_host']) . "');
define('SMTP_PORT', " . (isset($admin_config['smtp_port']) ? (int)$admin_config['smtp_port'] : 587) . ");
define('SMTP_USERNAME', '" . addslashes($admin_config['smtp_user']) . "');
define('SMTP_PASSWORD', '" . addslashes($admin_config['smtp_pass']) . "');
define('SMTP_FROM_EMAIL', '" . addslashes($admin_config['smtp_user']) . "');
define('SMTP_FROM_NAME', 'Promed Solutions Team');

// File Upload Configuration
define('UPLOAD_PATH', __DIR__ . '/../uploads/');
define('UPLOAD_URL', '/api/uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_EXTENSIONS', ['jpg', 'jpeg', 'png', 'gif', 'webp']);
?>";
    
    $configFile = __DIR__ . '/api/config/env.php';
    if (!file_put_contents($configFile, $config_content)) {
        throw new Exception('Failed to create configuration file. Please check file permissions.');
    }
}

function createDatabaseTables() {
    $db_config = $_SESSION['db_config'];
    
    $dsn = "mysql:host={$db_config['db_host']};dbname={$db_config['db_name']};charset=utf8mb4";
    $pdo = new PDO($dsn, $db_config['db_user'], $db_config['db_pass']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Read SQL file
    $sqlFile = __DIR__ . '/install/database.sql';
    if (file_exists($sqlFile)) {
        $sql = file_get_contents($sqlFile);
        // Remove comments and execute statements
        $statements = array_filter(array_map('trim', explode(';', $sql)));
        foreach ($statements as $statement) {
            if (!empty($statement) && !preg_match('/^--/', $statement)) {
                try {
                    $pdo->exec($statement);
                } catch (PDOException $e) {
                    // Ignore errors for existing tables
                    if (strpos($e->getMessage(), 'already exists') === false) {
                        error_log("SQL Error: " . $e->getMessage());
                    }
                }
            }
        }
    } else {
        // Create tables manually if SQL file doesn't exist
        $queries = [
            "CREATE TABLE IF NOT EXISTS services (
                id INT AUTO_INCREMENT PRIMARY KEY,
                first_name VARCHAR(255) NOT NULL,
                last_name VARCHAR(255) NOT NULL,
                company_name VARCHAR(255),
                email VARCHAR(255),
                phone_number VARCHAR(20),
                part VARCHAR(255) NOT NULL,
                modality VARCHAR(255) NOT NULL,
                manufacturer VARCHAR(255) NOT NULL,
                part_description TEXT,
                submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            "CREATE TABLE IF NOT EXISTS ser (
                id INT AUTO_INCREMENT PRIMARY KEY,
                first_name VARCHAR(255) NOT NULL,
                last_name VARCHAR(255) NOT NULL,
                company_name VARCHAR(255),
                email VARCHAR(255),
                phone_number VARCHAR(20),
                part_description TEXT NOT NULL,
                submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            "CREATE TABLE IF NOT EXISTS rental (
                id INT AUTO_INCREMENT PRIMARY KEY,
                first_name VARCHAR(255) NOT NULL,
                last_name VARCHAR(255) NOT NULL,
                company_name VARCHAR(255),
                email VARCHAR(255),
                phone_number VARCHAR(20),
                rental_service VARCHAR(255) NOT NULL,
                part_description TEXT,
                submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            "CREATE TABLE IF NOT EXISTS parts (
                id INT AUTO_INCREMENT PRIMARY KEY,
                part_number VARCHAR(255) NOT NULL,
                manufacturer VARCHAR(255) NOT NULL,
                modality VARCHAR(255) NOT NULL,
                product VARCHAR(255) NOT NULL,
                modal VARCHAR(255),
                image VARCHAR(500) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                KEY idx_modality (modality),
                KEY idx_manufacturer (manufacturer)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
        ];
        
        foreach ($queries as $query) {
            try {
                $pdo->exec($query);
            } catch (PDOException $e) {
                error_log("Error creating table: " . $e->getMessage());
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Infinity Medical - Installation Wizard</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; padding: 20px; }
        .container { max-width: 800px; margin: 0 auto; background: white; border-radius: 15px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); overflow: hidden; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 40px; text-align: center; }
        .header h1 { font-size: 2.5em; margin-bottom: 10px; }
        .header p { font-size: 1.2em; opacity: 0.9; }
        .content { padding: 40px; }
        .steps { display: flex; justify-content: space-between; margin-bottom: 40px; position: relative; }
        .steps::before { content: ''; position: absolute; top: 20px; left: 0; right: 0; height: 2px; background: #e0e0e0; z-index: 0; }
        .step { flex: 1; text-align: center; position: relative; z-index: 1; }
        .step-circle { width: 40px; height: 40px; border-radius: 50%; background: #e0e0e0; color: #999; display: flex; align-items: center; justify-content: center; margin: 0 auto 10px; font-weight: bold; transition: all 0.3s; }
        .step.active .step-circle { background: #667eea; color: white; }
        .step.completed .step-circle { background: #4CAF50; color: white; }
        .step-label { font-size: 0.9em; color: #666; }
        .step.active .step-label { color: #667eea; font-weight: bold; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; margin-bottom: 8px; color: #333; font-weight: 500; }
        .form-group input, .form-group select { width: 100%; padding: 12px; border: 2px solid #e0e0e0; border-radius: 8px; font-size: 1em; transition: border-color 0.3s; }
        .form-group input:focus, .form-group select:focus { outline: none; border-color: #667eea; }
        .grid-2 { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; }
        .btn { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 15px 30px; border: none; border-radius: 8px; font-size: 1em; font-weight: bold; cursor: pointer; transition: transform 0.2s, box-shadow 0.2s; width: 100%; }
        .btn:hover { transform: translateY(-2px); box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4); }
        .btn:active { transform: translateY(0); }
        .alert { padding: 15px; border-radius: 8px; margin-bottom: 20px; }
        .alert-error { background: #ffebee; color: #c62828; border: 1px solid #ef5350; }
        .alert-success { background: #e8f5e9; color: #2e7d32; border: 1px solid #4CAF50; }
        .alert-info { background: #e3f2fd; color: #1565c0; border: 1px solid #42a5f5; }
        .info-box { background: #f5f5f5; padding: 20px; border-radius: 8px; margin-bottom: 20px; border-left: 4px solid #667eea; }
        .info-box h3 { margin-bottom: 10px; color: #333; }
        .info-box ul { margin-left: 20px; }
        .info-box li { margin-bottom: 5px; color: #666; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏥 Infinity Medical</h1>
            <p>Professional Installation Wizard</p>
        </div>
        
        <div class="content">
            <!-- Progress Steps -->
            <div class="steps">
                <div class="step <?= $step >= 1 ? ($step > 1 ? 'completed' : 'active') : '' ?>">
                    <div class="step-circle">1</div>
                    <div class="step-label">Welcome</div>
                </div>
                <div class="step <?= $step >= 2 ? ($step > 2 ? 'completed' : 'active') : '' ?>">
                    <div class="step-circle">2</div>
                    <div class="step-label">Database</div>
                </div>
                <div class="step <?= $step >= 3 ? ($step > 3 ? 'completed' : 'active') : '' ?>">
                    <div class="step-circle">3</div>
                    <div class="step-label">Admin Setup</div>
                </div>
                <div class="step <?= $step >= 4 ? 'active' : '' ?>">
                    <div class="step-circle">4</div>
                    <div class="step-label">Complete</div>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">❌ <?= htmlspecialchars($error) ?></div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    ✅ <?= htmlspecialchars($success) ?>
                    <br><br>
                    <strong>Your website is ready!</strong><br>
                    Redirecting to your website in 3 seconds...
                    <script>setTimeout(function(){ window.location.href = 'index.html'; }, 3000);</script>
                </div>
            <?php endif; ?>

            <?php
            switch($step):
                case 1:
            ?>
                <h2 style="margin-bottom: 20px;">Welcome to Infinity Medical</h2>
                <p style="color: #666; margin-bottom: 30px;">
                    This installation wizard will help you set up your medical equipment website in just a few simple steps.
                </p>
                
                <div class="info-box">
                    <h3>✅ What You'll Get:</h3>
                    <ul>
                        <li>Complete medical equipment catalog system</li>
                        <li>Contact forms with email notifications</li>
                        <li>Rental service management</li>
                        <li>Admin panel for parts management</li>
                        <li>Fully responsive design</li>
                        <li>Secure authentication system</li>
                    </ul>
                </div>

                <div class="alert alert-info">
                    <strong>📋 Requirements:</strong> PHP 7.4+, MySQL 5.7+, mod_rewrite enabled, PDO extension
                </div>

                <a href="?step=2" class="btn" style="text-decoration: none; display: block; text-align: center;">Start Installation →</a>
            <?php break; case 2: ?>
                <h2 style="margin-bottom: 20px;">📊 Database Configuration</h2>
                <p style="color: #666; margin-bottom: 30px;">Enter your MySQL database credentials. The installer will create the database if it doesn't exist.</p>
                
                <form method="POST" class="space-y-4">
                    <div class="grid-2">
                        <div class="form-group">
                            <label>Database Host *</label>
                            <input type="text" name="db_host" value="localhost" required>
                        </div>
                        <div class="form-group">
                            <label>Database Name *</label>
                            <input type="text" name="db_name" placeholder="infinity_medical" required>
                        </div>
                    </div>
                    
                    <div class="grid-2">
                        <div class="form-group">
                            <label>Database Username *</label>
                            <input type="text" name="db_user" required>
                        </div>
                        <div class="form-group">
                            <label>Database Password *</label>
                            <input type="password" name="db_pass" required>
                        </div>
                    </div>
                    
                    <div class="alert alert-info">
                        <strong>💡 Tip:</strong> You can find these details in your cPanel → MySQL Databases section.
                    </div>
                    
                    <button type="submit" class="btn">Test Connection & Continue →</button>
                </form>
            <?php break; case 3: ?>
                <h2 style="margin-bottom: 20px;">👤 Admin & Email Setup</h2>
                <p style="color: #666; margin-bottom: 30px;">Create your admin account and configure email settings for form notifications.</p>
                
                <form method="POST">
                    <div class="info-box" style="margin-bottom: 20px;">
                        <h3>Admin Account</h3>
                        <div class="grid-2" style="margin-top: 15px;">
                            <div class="form-group">
                                <label>Admin Username *</label>
                                <input type="text" name="admin_user" value="admin" required>
                            </div>
                            <div class="form-group">
                                <label>Admin Password *</label>
                                <input type="password" name="admin_pass" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="info-box">
                        <h3>Email Configuration</h3>
                        <div class="form-group" style="margin-top: 15px;">
                            <label>SMTP Host *</label>
                            <input type="text" name="smtp_host" value="mail.infinitymedical.co" required>
                        </div>
                        <div class="grid-2">
                            <div class="form-group">
                                <label>Email Address *</label>
                                <input type="email" name="smtp_user" placeholder="noreply@infinitymedical.co" required>
                            </div>
                            <div class="form-group">
                                <label>Email Password *</label>
                                <input type="password" name="smtp_pass" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>SMTP Port</label>
                            <input type="number" name="smtp_port" value="587">
                        </div>
                    </div>
                    
                    <button type="submit" class="btn">Continue to Installation →</button>
                </form>
            <?php break; case 4: ?>
                <h2 style="margin-bottom: 20px;">🚀 Final Installation</h2>
                
                <?php if (!$success): ?>
                    <p style="color: #666; margin-bottom: 30px;">Ready to complete the installation? This will:</p>
                    <div class="info-box">
                        <ul>
                            <li>Create configuration files</li>
                            <li>Set up database tables</li>
                            <li>Create necessary directories</li>
                            <li>Configure security settings</li>
                        </ul>
                    </div>
                    
                    <form method="POST">
                        <button type="submit" class="btn">Complete Installation 🎉</button>
                    </form>
                <?php else: ?>
                    <div class="alert alert-success">
                        <h3 style="margin-bottom: 15px;">✅ Installation Successful!</h3>
                        <p style="margin-bottom: 15px;">Your Infinity Medical website is now ready to use.</p>
                        <div style="margin-top: 20px;">
                            <a href="index.html" class="btn" style="text-decoration: none; display: block; text-align: center; margin-bottom: 10px;">Visit Your Website</a>
                            <p style="text-align: center; color: #666; font-size: 0.9em;">
                                Admin Panel: Use the credentials you created to login
                            </p>
                        </div>
                    </div>
                <?php endif; ?>
            <?php break;
            endswitch;
            ?>
        </div>
    </div>
</body>
</html>
